/*
 *  main.c
 *  ConvertOanda
 *
 *  Created by Eckhard Lehmann on 21.11.08.
 *  Copyright 2008 __MyCompanyName__. All rights reserved.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

// Min/Max
#define min(a,b) ((a) < (b) ? (a) : (b))
#define max(a,b) ((a) > (b) ? (a) : (b))

// Time structure for manipulating time values
struct tm tm;

// Time of next operation
time_t gmt;

// Tick data
FILE * tick_file;      // FXTicks CSV file handle
int    tick_count = 0; // Number of ticks read
time_t tick_gmt;       // GMT of current tick
float  tick_bid;       // Bid price of current tick
float  tick_ask;       // Ask price of current tick

int gPrintTime = 1;

// OHLC data
typedef struct 
{
    FILE *file;      // CSV file handle
    char *pair;
    int    count;     // Number of lines
    time_t start_gmt; // First GMT for time period
    time_t end_gmt;   // Last GMT for time period
    float  open_bid;  // Opening Bid price
    float  open_ask;  // Opening Ask price
    float  high_bid;  // High Bid price
    float  high_ask;  // High Ask price
    float  low_bid;   // Low Bid price
    float  low_ask;   // Low Ask price
    float  close_bid; // Closing Bid price
    float  close_ask; // Closing Ask price
} ohlc_struct;

ohlc_struct minute_ohlc;
ohlc_struct hour_ohlc;
ohlc_struct day_ohlc;

// Log output
time_t stdout_gmt;

// Open OHLC file (oanda_<pair><suffix>) and initialise variables
void
open_ohlc (ohlc_struct * ohlc, char * pair, char * suffix, time_t period)
{
    char * filename = calloc (6 + strlen (pair) + strlen (suffix) + 1, 1);
    strcpy (filename, "oanda_");
    strcat (filename, pair);
    strcat (filename, suffix);
    ohlc->file = fopen (filename, "w");
    if (ohlc->file == NULL)
    {
        printf ("Unable to open %s for output\n", filename);
        exit (0);
    }
    ohlc->pair = pair;
    ohlc->count = 0;
    ohlc->start_gmt = timegm (&tm);
    ohlc->end_gmt = ohlc->start_gmt + period - 1;
    ohlc->open_bid = 0.0;
    ohlc->open_ask = 0.0;
    ohlc->close_bid = ohlc->high_bid = ohlc->low_bid = tick_bid;
    ohlc->close_ask = ohlc->high_ask = ohlc->low_ask = tick_ask;
    
    fprintf(ohlc->file, "<TICKER>,<DTYYYYMMDD>,<TIME>,<OPEN>,<HIGH>,<LOW>,<CLOSE>\n");
    
    free (filename);
}

// Read tick into variables
int read_tick ()
{
    if (8 != fscanf (tick_file, "%d/%d/%d %d:%d:%d,%f,%f\n", 
                     &tm.tm_mday, &tm.tm_mon, &tm.tm_year, &tm.tm_hour, &tm.tm_min, &tm.tm_sec, 
                     &tick_bid, &tick_ask)) {
        return 0;
    }
    
    tm.tm_mon -= 1;
    tm.tm_year += 100;
    tm.tm_wday = 0;
    tm.tm_yday = 0;
    tm.tm_isdst = 0;
    tm.tm_gmtoff = 0;
    tm.tm_zone  = "UTC";
    
    tick_gmt = timegm (&tm);
    tick_count += 1;
    
    return 1;
}

// Update OHLC with tick data
void
update_ohlc (ohlc_struct * ohlc)
{
    if (tick_gmt == ohlc->start_gmt)
    {
        ohlc->close_bid = ohlc->high_bid = ohlc->low_bid = ohlc->open_bid = tick_bid;
        ohlc->close_ask = ohlc->high_ask = ohlc->low_ask = ohlc->open_ask = tick_ask;
    }
    else
    {
        ohlc->close_bid = tick_bid;
        ohlc->close_ask = tick_ask;
        ohlc->high_bid  = max (tick_bid, ohlc->high_bid);
        ohlc->high_ask  = max (tick_ask, ohlc->high_ask);
        ohlc->low_bid   = min (tick_bid, ohlc->low_bid);
        ohlc->low_ask   = min (tick_ask, ohlc->low_ask);
    }
}

// Write OHLC and reset data
void
write_ohlc (ohlc_struct * ohlc)
{
    time_t period;
    
    period = ohlc->end_gmt - ohlc->start_gmt + 1;
    if (ohlc->close_bid != 0.0)
    {
        if (ohlc->open_bid != 0.0)
        {
            gmtime_r (&ohlc->start_gmt, &tm);
            if (gPrintTime) {
                time_t dnTime = timegm(&tm);
                double dnval = (dnTime / 86400.0) + 719529;
                printf("Time: %d - %s - %15.5f \n", dnTime, asctime(&tm), dnval);
                gPrintTime = 0;
            }
            
            /*fprintf (ohlc->file, 
                     "%04d%02d%02d%02d%02d%02d,%3.5f,%3.5f,%3.5f,%3.5f,%3.5f,%3.5f,%3.5f,%3.5f\n",
                     tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tm.tm_hour, tm.tm_min, tm.tm_sec,
                     ohlc->open_bid, ohlc->open_ask, ohlc->high_bid, ohlc->high_ask,
                     ohlc->low_bid, ohlc->low_ask, ohlc->close_bid, ohlc->close_ask);*/
            fprintf(ohlc->file, "%s,%04d%02d%02d,%02d%02d%02d,%3.5f,%3.5f,%3.5f,%3.5f\n", ohlc->pair, 
                    tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday,
                    tm.tm_hour, tm.tm_min, tm.tm_sec,
                    (ohlc->open_bid + ohlc->open_ask) / 2,
                    (ohlc->high_bid + ohlc->high_ask) / 2,
                    (ohlc->low_bid + ohlc->low_ask) / 2,
                    (ohlc->close_bid + ohlc->close_ask) / 2);
            /*fprintf(ohlc->file, "%.5f,%3.5f,%3.5f,%3.5f,%3.5f\n", 
                    (timegm(&tm) / 86400.0) + 719529,
                    (ohlc->open_bid + ohlc->open_ask) / 2,
                    (ohlc->high_bid + ohlc->high_ask) / 2,
                    (ohlc->low_bid + ohlc->low_ask) / 2,
                    (ohlc->close_bid + ohlc->close_ask) / 2);*/
            fflush (ohlc->file);
            ohlc->count += 1;
        }
        ohlc->high_bid = ohlc->low_bid = ohlc->open_bid = ohlc->close_bid;
        ohlc->high_ask = ohlc->low_ask = ohlc->open_ask = ohlc->close_ask;
        ohlc->close_bid = 0.0;
        ohlc->close_ask = 0.0;
    }
    ohlc->start_gmt += period;
    ohlc->end_gmt   += period;
}

// Main program
int
main (int argc, char *argv[])
{
    // Check number of arguments
    if (argc != 3)
    {
        printf ("Usage: fxticks2mysql csv-in pair\n");
        printf ("       (generates 3 csv output files - minute, hour and day)\n");
        exit (0);
    }
    
    // Open tick input file
    tick_file = fopen (argv[1], "r");
    if (tick_file == NULL)
    {
        printf ("Unable to open %s for input\n", argv[1]);
        exit (0);
    }
    
    // Read first tick
    if (read_tick () == 0)
    {
        printf ("Cannot read first line from %s\n", argv[1]);
        exit (0);
    };
    
    // Open ohlc files and initialise data structures
    tm.tm_sec = 0;
    open_ohlc (&minute_ohlc, argv[2], "_minute.csv", 60);
    tm.tm_min = 0;
    open_ohlc (&hour_ohlc, argv[2], "_hour.csv", 3600);
    tm.tm_hour = 0;
    open_ohlc (&day_ohlc, argv[2], "_day.csv", 86400);
    
    // Standard output GMT
    stdout_gmt = timegm (&tm) + 86399;
    
    // Loop through GMT values
    while (1)
    {
        // Determine next gmt to process
        gmt = min (stdout_gmt, min (tick_gmt, min (minute_ohlc.end_gmt, min (hour_ohlc.end_gmt, day_ohlc.end_gmt))));
        
        // Update ohlc prices
        if (gmt == tick_gmt)
        {
            update_ohlc (&minute_ohlc);
            update_ohlc (&hour_ohlc);
            update_ohlc (&day_ohlc);
            if (read_tick () == 0) {
                break;
            }
        }
        
        // End of minute, hour or day
        if (gmt == minute_ohlc.end_gmt) {
            write_ohlc (&minute_ohlc);
        }
        if (gmt == hour_ohlc.end_gmt) {
            write_ohlc (&hour_ohlc);
        }
        if (gmt == day_ohlc.end_gmt) {
            write_ohlc (&day_ohlc);
        }
        
        // Day log
        if (gmt == stdout_gmt)
        {
            gmtime_r (&gmt, &tm);
            printf ("%04d-%02d-%02d: %d ticks, %d minutes, %d hours, %d days\n", 
                    tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tick_count, minute_ohlc.count, hour_ohlc.count, day_ohlc.count);
            stdout_gmt += 86400;
        }
    }
    
    // Write final lines
    if (tick_gmt >= minute_ohlc.start_gmt) {
        write_ohlc (&minute_ohlc);
    }
    if (tick_gmt >= hour_ohlc.start_gmt) {
        write_ohlc (&hour_ohlc);
    }
    if (tick_gmt >= day_ohlc.start_gmt) {
        write_ohlc (&day_ohlc);
    }
    
    // Final day log
    gmtime_r (&gmt, &tm);
    printf ("%04d-%02d-%02d: %d ticks, %d minutes, %d hours, %d days\n", 
            tm.tm_year + 1900, tm.tm_mon + 1, tm.tm_mday, tick_count, minute_ohlc.count, hour_ohlc.count, day_ohlc.count);
    
    // Close files
    fclose (tick_file);
    fclose (minute_ohlc.file);
    fclose (hour_ohlc.file);
    fclose (day_ohlc.file);
    
    exit (1);
}