#! /bin/bash

# functions to convert gregorian dates to and from julian day numbers are adopted
# from javascript at <http://www.fourmilab.ch/documents/calendar/>

GREGORIAN_EPOCH=1721425   # precise: 1721425.5; but bash can do only integer arithmetic

function leap_year   # (year)
{ echo $(( $1 % 4 == 0 && ( ! ( $1 % 100 == 0 && $1 % 400 != 0 ) ) ))
}

function gregorian_date_to_julian_day   # ($1 = year, $2 = month, $3 = day)
{ echo $((
    GREGORIAN_EPOCH - 1
  + ( $1 - 1 ) * 365
  + ( $1 - 1 ) / 4
  - ( $1 - 1 ) / 100
  + ( $1 - 1 ) / 400
  + ( 367 * $2 - 362 ) / 12
  + ( $2 <= 2 ? 0 : $( leap_year $1 ) ? -1 : -2 )
  + $3
  ))
}

function julian_day_to_gegorian_date   # ($1 = julian_day)
{ local     wjd=$1   # precise: floor($1 -0.5) + 0.5; but bash can only do integer arithmetic
  local  depoch=$(( wjd    - GREGORIAN_EPOCH ))
  local   qcent=$(( depoch / 146097 ))
  local     dqc=$(( depoch % 146097 ))
  local    cent=$(( dqc    /  36524 ))
  local   dcent=$(( dqc    %  36524 ))
  local    quad=$(( dcent  /   1461 ))
  local   dquad=$(( dcent  %   1461 ))
  local  yindex=$(( dquad  /    365 ))
  local       y=$(( qcent * 400 + cent * 100 + quad * 4 + yindex ))

  if (( ! ( cent == 4  || yindex == 4 ) )); then y=$(( ++ y )); fi

  local yearday=$(( wjd - $( gregorian_date_to_julian_day y 1 1 ) ))
  local leapadj=$(( wjd < $( gregorian_date_to_julian_day y 3 1 ) ? 0 : $( leap_year y ) ? 1 : 2 ))
  local       m=$(( ( ( yearday + leapadj ) * 12 + 373 ) / 367 ))
  local       d=$(( wjd - $( gregorian_date_to_julian_day y m 1 ) + 1 ))

  if (( m < 10 )); then m=0$m; fi
  if (( d < 10 )); then d=0$d; fi

  echo $y $m $d
}

function julian_date_week_day   # ($1 = julian_day)
{ echo $(( ( $1 + 2 ) % 7 ))    # precise: + 1.5; but bash can only do integer arithmetic
}

function ask_date   # ($1 = date kind (Start/End), $2 = variable)
{ local ok=0

  while (( ! ok ))
  do
    read -p "$1"' day (YYYY-MM-DD): ' $2
    if [[ ${!2} =~ [0-9]{4}-[0-9]{2}-[0-9]{2} ]]
    then ok=1
    else echo "$1 date has not the format (YYYY-MM-DD), reenter"
    fi
  done
}

function ask_period
{ until [[ $REPLY == 'y' ]]
  do
    ask_date Start START_DATE
    ask_date End     END_DATE

    START_JULIAN_DAY=$( gregorian_date_to_julian_day ${START_DATE:0:4} ${START_DATE:5:2} ${START_DATE:8:2} )
      END_JULIAN_DAY=$( gregorian_date_to_julian_day   ${END_DATE:0:4}   ${END_DATE:5:2}   ${END_DATE:8:2} )

    if (( $START_JULIAN_DAY > $END_JULIAN_DAY ))
    then
      echo "Start date ($START_DATE) can't be bigger than end date ($END_DATE), reenter"
      continue
    fi

    read -p "Is period form $START_DATE to $END_DATE correct? (y/n) "
  done
}

function get_one_quote_file   # ($1 = year, $2 = month, $3 = day)
{ local  interval=60
  local    symbol=25
  local    output="DAXI.$1-$2-$3.1 min.csv"
  local      site=www.dukascopy.com
  local file_name="/freeApplets/exp/exp.php?fromD=$2.$3.$1&np=1000&interval=$interval&DF=d.m.Y&Stock=$symbol&endSym=win&split=tz"
  local   referer=http://$site/freeApplets/exp/

  gawk -f- > "$output" <<____EOS__

    BEGIN {

      webSite    = "/inet/tcp/0/$site/http"
      httpHeader = "\
GET $file_name HTTP/1.1\n\
Host: $site\n\
User-Agent: Mozilla/5.0 (Windows; U; Windows NT 5.1; de; rv:1.9.2.13) Gecko/20101203 Firefox/3.6.13 ( .NET CLR 3.5.30729; .NET4.0C)\n\
Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8\n\
Accept-Language: de-de,de;q=0.8,en-us;q=0.5,en;q=0.3\n\
Accept-Encoding: gzip,deflate\n\
Accept-Charset: ISO-8859-1,utf-8;q=0.7,*;q=0.7\n\
Keep-Alive: 115\n\
Connection: keep-alive\n\
Referer: $referer\n\
\n\
"
      print httpHeader |& webSite
      content = 0
      while ( (webSite |& getline) > 0 )
        if ( content ) print \$0
        else if ( \$0 == "\r" ) content = 1
      close(webSite)
    }

____EOS__
}

function get_quote_files
{ local  sleepMin=60
  local  sleepMax=120

  for (( day = $START_JULIAN_DAY; day <= $END_JULIAN_DAY; ++ day ))
  do
    local weekDay=$( julian_date_week_day $day )
    if [[ $weekDay == 0 || $weekDay == 6 ]]; then continue; fi
    local gregorianDate=$( julian_day_to_gegorian_date $day )
    local gregorianDateArray=( $gregorianDate )
    echo Downloading quotes from date ${gregorianDateArray[0]}-${gregorianDateArray[1]}-${gregorianDateArray[2]}
    get_one_quote_file $gregorianDate
    local sleepTime=$(( sleepMin + $RANDOM * ( sleepMax - sleepMin ) / 32767 ))
    if (( day < $END_JULIAN_DAY ))
      then
        echo "Waiting now $sleepTime seconds ..."
        sleep $sleepTime
    fi
  done
}

ask_period
get_quote_files
