//+------------------------------------------------------------------+
//|                                              Modify_Position.mq5 |
//|                                                          private |
//|                                                                  |
//+------------------------------------------------------------------+
#property copyright "private"
#property link      ""
#property version   "1.00"

#include <ChartObjects\ChartObjectsLines.mqh>

enum ENUM_DIRECTION{
   TRADE_LONG=1, //Long
   TRADE_SHORT=-1 //Short
};

ENUM_DIRECTION eDirection;

CChartObjectHLine oLine[2];
bool bInitCheckFailed;

//--- input parameters
input double   cdNewSL=0.0; //New SL (0..unchanged)
input double   cdNewTP=0.0; //New TP (0..unchanged)
input string csNotBefore="0:00"; //Do not submit Change before (hh:mm)
input string csNotAfter="23:59"; //Do not submit Change after (hh:mm)



int ExitAfterError(string message)
{
   bInitCheckFailed=true;
   MessageBox(message,"Operation failed", MB_ICONERROR);
   //ExpertRemove();
   Comment("Check Parameters!");
   return 0;
}

int CommentAndExit(string message)
{
   Comment(message);
   ExpertRemove();
   return 0;
}

void WarningMessage(string message)
{
   MessageBox(message,"Warning!", MB_ICONEXCLAMATION);
}

ENUM_DIRECTION GetPositionDirection()
{
   return (PositionGetInteger(POSITION_TYPE)==POSITION_TYPE_BUY)?TRADE_LONG:TRADE_SHORT;
}

double GetOldSL()
{
   return PositionGetDouble(POSITION_SL);
}

double GetOldTP()
{
   return PositionGetDouble(POSITION_TP);
}


//+------------------------------------------------------------------+
//| Expert initialization function                                   |
//+------------------------------------------------------------------+
int OnInit()
{
datetime dtNotBefore,dtNotAfter;
double dOldSL,dOldTP;
//---
   bInitCheckFailed=false;
   
   //check for open position and direction
   Sleep(1000); //after MT5 restart open positions are not loaded immediately, failing the init check below
   if (PositionSelect(Symbol()))
   {
      eDirection=GetPositionDirection();
      dOldSL=GetOldSL();
      dOldTP=GetOldTP();
   }else{
      if (GetLastError()==ERR_TRADE_POSITION_NOT_FOUND)
         ExitAfterError("Position not found.");
      else
         ExitAfterError("Error while checking open position.");
      ExpertRemove();
      return 0;
   }   
   
   //Plausibility Checks
   if ((cdNewSL==0)&&(cdNewTP==0))
      return ExitAfterError("No Change requested.");
   
   dtNotBefore=StringToTime(csNotBefore);   
   dtNotAfter=StringToTime(csNotAfter);   
   if (!(dtNotBefore<dtNotAfter))
      return ExitAfterError("No submission time frame allowed");
      
      

   
   if ((cdNewSL!=0)&&((dOldSL-cdNewSL)*eDirection>0))
   {
      WarningMessage("You are increasing the SL distance. This increases the risk of this trade. Make sure you know what you are doing.");
   }
   if ((cdNewTP!=0)&&((dOldTP-cdNewTP)*eDirection>0))
   {
      WarningMessage("You are decreasing the TP distance.");
   }
   
   Comment("Change Open Position");
   if (cdNewSL>0)
   {
      oLine[1].Create(0,"New SL",0,cdNewSL);
      oLine[1].Color(eDirection==TRADE_SHORT?clrLightGreen:clrLightSalmon);   
   }
   if (cdNewTP>0)
   {
      oLine[0].Create(0,"New TP",0,cdNewTP);
      oLine[0].Color(eDirection==TRADE_SHORT?clrLightGreen:clrLightSalmon);     
   }
//---
   return(INIT_SUCCEEDED);
}
//+------------------------------------------------------------------+
//| Expert deinitialization function                                 |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
  {
//---
   
  }
//+------------------------------------------------------------------+
//| Expert tick function                                             |
//+------------------------------------------------------------------+
void OnTick()
{
datetime dtNotBefore,dtNotAfter,dtNow;
double dUseSL,dUseTP,dOld;
double dCurrentPrice,dMinOrderDistance;
bool bTrailingChange;
MqlTradeRequest request={0};
MqlTradeResult result={0};
//---
   if (bInitCheckFailed)
   {
      Comment("Check Parameters!");
      return;
   }

   //check submission time frame
   dtNow=TimeCurrent();
   dtNotBefore=StringToTime(csNotBefore);   
   dtNotAfter=StringToTime(csNotAfter);
   if (dtNow<dtNotBefore) return; //continue later
   if (dtNow>dtNotAfter)
   {
      CommentAndExit("Submission time missed at: "+TimeToString(dtNow));
      return;     
   }
   
   //get Position Info
   if (PositionSelect(Symbol()))
   {
      eDirection=GetPositionDirection();
      dUseSL=GetOldSL();
      dUseTP=GetOldTP();
   }else{
      Comment("Error while checking open position. "+TimeToString(dtNow)); //probably no position opened!
      if (GetLastError()==ERR_TRADE_POSITION_NOT_FOUND)
         ExpertRemove(); //do not try again
      return; //try again later
   }      
   if ((dUseSL<=0)||(dUseTP<=0))
   {
      Comment("Current SL or TP not allowed. This is a bug.");
      return; //try again later
   }
   
   //calculate everything required
   dCurrentPrice=SymbolInfoDouble(Symbol(),(eDirection==TRADE_LONG)?SYMBOL_BID:SYMBOL_ASK); //as we deal with closing positions (SL/TP) a long trade corresponds to bid price
   dMinOrderDistance=SymbolInfoDouble(Symbol(),SYMBOL_POINT)*SymbolInfoInteger(Symbol(),SYMBOL_TRADE_STOPS_LEVEL);
    
   bTrailingChange=false;
   //calculate new TP and SL
   if (cdNewSL>0)
   {
      dOld=dUseSL;
      dUseSL=SymbolInfoDouble(Symbol(),SYMBOL_TRADE_TICK_SIZE)*NormalizeDouble(cdNewSL/SymbolInfoDouble(Symbol(),SYMBOL_TRADE_TICK_SIZE),0);
;
      if (((dCurrentPrice-dUseSL)*eDirection)<dMinOrderDistance)
      {
         dUseSL=dCurrentPrice-dMinOrderDistance*eDirection;      
         //if old SL/TP was already closer, dont do anything
         if ((dOld-dUseSL)*eDirection>0)
            return;
         bTrailingChange=true;
      }
   }
   if (cdNewTP>0)
   {
      dOld=dUseTP;
      dUseTP=SymbolInfoDouble(Symbol(),SYMBOL_TRADE_TICK_SIZE)*NormalizeDouble(cdNewTP/SymbolInfoDouble(Symbol(),SYMBOL_TRADE_TICK_SIZE),0);
      if (((dUseTP-dCurrentPrice)*eDirection)<dMinOrderDistance)
      {
         dUseTP=dCurrentPrice+dMinOrderDistance*eDirection;
         //if old SL/TP was already closer, dont do anything
         if ((dUseTP-dOld)*eDirection>0)
            return;         
         bTrailingChange=true;
      }
   }   
   
   //create request
   request.action=TRADE_ACTION_SLTP;
   request.symbol=Symbol();
   request.sl=dUseSL;
   request.tp=dUseTP;
   
   if (OrderSend(request,result))
   {
      if (bTrailingChange)
      {
         Comment("Position modified with SL/TP adjusted to minimum stop level distance. "+TimeToString(dtNow));
         return;
      }else{
         CommentAndExit("Position modified. Exiting now...");
         return;
      }
   }else{
      Comment("Position modification failed: "+result.comment);
   }
      
            
}
//+------------------------------------------------------------------+
