//+------------------------------------------------------------------+
//|                                                   OandaStyle.mq4 |
//|                                                                  |
//|                                                                  |
//+------------------------------------------------------------------+
#property copyright "Copyright Shimodax"
#property link      "http://www.strategybuilderfx.com"

#property indicator_chart_window

/* Introduction: 

   Indicator to show current and previous orders similar to
   the Oanda trading applet.  

   Trade entries will be shown with golden arrows.  Current
   TP/SL levels will be shown as red and green lines.  
   If you show historical trades (closed), a red closing arrow
   will indicate a loss, green closing arrows will indicate
   profit.
   
   Important: After placing a trade the order will be shown
   on screen only after a new price tick occurred!
   
   
   Recommended chart setup to duplicate Oanda (Chart-Options, F8) is:
   - Background: Black
   - Foreground: White
   - Grid: Black
   - Bar Up/Down: DarkGray
   - BullCandle: Black
   - BearCandle: DarkGray
   - LineGraph: DarkGray
   - Volumes: Green or Red
   - Ask-Line: Black
   - Stop-Levels: Black

   - Show Ask-Line: off
   - Show Grid: off
   
   Program Options (Tools-Menu, Options, Chart):
   - Show Trade-Levels: off
   - Show Ask-Line: off
      

   Enjoy!
   
   Markus

*/

extern bool ShowOpenOrders= true;
extern bool ShowClosedOrders= true;
extern bool ShowLimitOrders= true;
extern bool ShowBidAskLines= true;

extern color ColorBuySymbol= Gold;
extern color ColorSellSymbol= Gold;
extern color ColorSL= Red;
extern color ColorTP= ForestGreen;

extern int LineStyleActive= STYLE_SOLID;
extern int LineStylePending= STYLE_DOT;

extern color ColorBidAsk= Khaki;
extern int LineStyleBidAsk= STYLE_DOT;




//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int init()
{
   return(0);
}


int deinit()
{
   int obj_total= ObjectsTotal();
   
   for (int i= obj_total; i>=0; i--) {
      string name= ObjectName(i);
    
      if (StringSubstr(name,0,7)=="[Trade]") 
         ObjectDelete(name);
   }
   
   return(0);
}


  
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
int start()
{
   static datetime timelastupdate= 0;
   static datetime lasttimeframe= 0;
   
    
   // no need to update these buggers too often   
   if (CurTime()-timelastupdate < 600 && Period()==lasttimeframe)
      return (0);
   
   deinit();  // delete all previous lines
      
   string type;
   color col, col2;
   int i, linestyle, op2, basetype;
   bool ignore;
   

   //
   // display open orders
   //
   for(i=0; i<OrdersTotal();i++) {
   
      if(OrderSelect(i,SELECT_BY_POS,MODE_TRADES)==false) 
         break;
         
      if(OrderSymbol()==Symbol()) {
      
         switch (OrderType()) {
            case OP_BUY: 
                           type= "Buy"; 
                           col= ColorBuySymbol; 
                           linestyle= LineStyleActive; 
                           basetype= 1;   // open
                           break;
                           
            case OP_SELL:  type= "Sell"; 
                           col= ColorSellSymbol; 
                           linestyle= LineStyleActive; 
                           basetype= 1;   // open
                           break;
            
            case OP_BUYLIMIT: 
            case OP_BUYSTOP:
                           type= "Buy"; 
                           col= ColorBuySymbol; 
                           linestyle= LineStylePending; 
                           basetype= 2;   // pending
                           break;

            
            case OP_SELLLIMIT: 
            case OP_SELLSTOP:
                           type= "Sell"; 
                           col= ColorSellSymbol; 
                           linestyle= LineStylePending; 
                           basetype= 2;   // pending
                           break;
         }
         
         
         if ((basetype==1 && !ShowOpenOrders) || (basetype==2 && !ShowLimitOrders))
            continue;
         
         SetArrow(type+" Open", OrderTicket(), OrderType(), Time[iBarShift(NULL,0, OrderOpenTime())], OrderOpenPrice(), col);
         SetLevel(type+" TP", OrderTicket(), MathMin(OrderOpenTime(), Time[10]), OrderTakeProfit(), ColorTP, linestyle, 1);
         SetLevel(type+" SL", OrderTicket(), MathMin(OrderOpenTime(), Time[10]), OrderStopLoss(),   ColorSL, linestyle, 1);
      }
   }



   //
   // display open orders
   //
   if (ShowClosedOrders) {
   
      for(i=0; i<HistoryTotal();i++) {
   
         if (OrderSelect(i,SELECT_BY_POS,MODE_HISTORY)==false) 
            break;
         
         if (OrderSymbol()==Symbol()) {
            ignore= false;

            switch (OrderType()) {
               case OP_BUY:   type= "Buy"; 
                              col= ColorBuySymbol; 
                              op2= OP_SELL;
                              break;
                           
               case OP_SELL:  type= "Sell"; 
                              col= ColorSellSymbol; 
                              op2= OP_BUY;
                              break;
                              
               case OP_BUYLIMIT: 
               case OP_BUYSTOP:
               case OP_SELLLIMIT: 
               case OP_SELLSTOP:  
                              ignore= true;   // cancelled (closed with stop/limit state)
                              
            }
         
            if (!ignore) {
               if (OrderProfit()>=0) 
                  col2= ColorTP; 
               else
                  col2= ColorSL;

               SetArrow(type+" Open", OrderTicket(), OrderType(), Time[iBarShift(NULL,0, OrderOpenTime())], OrderOpenPrice(), col);
               SetArrow(type+" Close "+DoubleToStr(OrderProfit(),2)+ "", OrderTicket(), op2, Time[iBarShift(NULL,0, OrderCloseTime())], OrderClosePrice(), col2);
           }
        }
      }
   }
   
   
   if (ShowBidAskLines) {
      SetHLevel("Bid", Bid, ColorBidAsk, LineStyleBidAsk, 1);
      SetHLevel("Ask", Ask, ColorBidAsk, LineStyleBidAsk, 1);
   }
   
   return(0);
}



//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void SetArrow(string text, int ticket, int type, datetime time, double level, color col1)
{
   if (level!=0) {
      bool issell= (type==OP_SELL || type==OP_SELLSTOP || type==OP_SELLLIMIT);

      int digits= Digits;
      double extra= 0;
      string linename= "[Trade] " + text + " "+ ticket,
             symbol; 

      if (type==OP_SELL || type==OP_BUY) {
         symbol= StringSetChar(" ", 0, '\xE9');
      }
      else 
      if (type==OP_SELLLIMIT || type==OP_SELLSTOP || type==OP_BUYLIMIT || type==OP_BUYSTOP) {
         symbol= StringSetChar(" ", 0, '\xF1');
      }

      if (!issell) 
         extra= Point;
      else
         extra= -Point;


      // create or move the horizontal line   
      if (ObjectFind(linename)!=0) {
         ObjectCreate(linename, OBJ_TEXT, 0, time, level + extra);
      }

      ObjectMove(linename, 0, time, level + extra);
      ObjectSetText(linename, symbol, 12, "Wingdings", col1);
         
      if (issell)
         ObjectSet(linename, OBJPROP_ANGLE, 180);
   }
}
      

//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void SetLevel(string text, int ticket, datetime time, double level, color col1, int linestyle, int thickness)
{
   if (level!=0) {
   
      int digits= Digits;
      string labelname= "[Trade] " + text + ticket + " Label",
             linename= "[Trade] " + text + ticket + " Limit",
             pricelabel; 

      // create or move the horizontal line   
      if (ObjectFind(linename) != 0) {
         ObjectCreate(linename, OBJ_TREND, 0, time, level, Time[0],level);
         ObjectSet(linename, OBJPROP_STYLE, linestyle);
         ObjectSet(linename, OBJPROP_COLOR, col1);
         ObjectSet(linename, OBJPROP_WIDTH, thickness);
      }
      else {
         ObjectMove(linename, 1, Time[0],level);
         ObjectMove(linename, 0, time, level);
      }
   

      // put a label on the line   
      if (ObjectFind(labelname) != 0) {
         ObjectCreate(labelname, OBJ_TEXT, 0, time, level);
      }
      else {
         ObjectMove(labelname, 0, time, level);
      }

      pricelabel= " " + text;
      if (StrToInteger(text)==0) 
         pricelabel= /* pricelabel + ": "+ */ DoubleToStr(level, Digits);
   
      ObjectSetText(labelname, pricelabel, 8, "Arial", White);
      
      if (level>(Bid+Ask)/2) 
         ObjectSet(labelname, OBJPROP_CORNER, 2);
      else
         ObjectSet(labelname, OBJPROP_CORNER, 0);
   }
}
 
 
 
//+------------------------------------------------------------------+
//|                                                                  |
//+------------------------------------------------------------------+
void SetHLevel(string text, double level, color col1, int linestyle, int thickness)
{
   int digits= Digits;
   string linename= "[Trade] " + text + " Line";

   // create or move the horizontal line   
   if (ObjectFind(linename) != 0) {
      ObjectCreate(linename, OBJ_HLINE, 0, 0, level);
   }

   ObjectSet(linename, OBJPROP_STYLE, linestyle);
   ObjectSet(linename, OBJPROP_COLOR, col1);
   ObjectSet(linename, OBJPROP_WIDTH, thickness);
   ObjectMove(linename, 0, Time[0], level);

}     

